module net.BurtonRadons.parse.parser;

private import net.BurtonRadons.parse.lexer;

/** Converts lexer streams of tokens into a parse tree. */

class Parser : Lexer
{
    import net.BurtonRadons.parse.declaration;
    import net.BurtonRadons.parse.expression;
    import net.BurtonRadons.parse.scope;
    import net.BurtonRadons.parse.statement;
    import net.BurtonRadons.parse.type;
    
    int parseErrorCount; /**< Number of parse errors that have occurred. */
    int semanticErrorCount; /**< Number of semantic errors that have occurred. */

    this (char [] filename)
    {
        super (filename);
    }

    this (char [] filename, ubyte [] data)
    {
        super (filename, data);
    }

    /** A parse error. */
    void parseError (MarkedError cause)
    {
        parseErrorCount ++;
        super.error (cause);
    }

    /** A semantic error. */
    void semanticError (MarkedError cause)
    {
        semanticErrorCount ++;
        super.error (cause);
    }

    /** Read in a module from the parser. */
    Module parseModule ()
    {
        Module mod = new Module (this);
        Declaration [] list;

        if (token.type == moduleToken)
            throw new Error ("TODO: 'module' unhandled.");
        if (!parseDeclarationList (list, eoiToken))
            return null;
        mod.list = (Statement []) list;
        return mod;
    }

    /** Parse everything up to and past the next semicolon. */
    void parseSemicolon ()
    {
        while (1)
        {
            if (nextToken ().type == eoiToken)
                return;
            if (token.type == semicolonToken)
            {
                nextToken ();
                return;
            }
        }
    }

    /** Parse a set of declarations, such as from the module or struct/class/union scopes. 
      * Stops when "finish" is encountered.
      */

    bit parseDeclarationList (inout Declaration [] list, TokenType finish)
    {
        while (1)
        {
            if (token.type == finish)
                return true;
            if (token.type == eoiToken)
            {
                expectToken (finish);
                return false;
            }
            if (!parseDeclaration (list))
                return false;
        }
    }

    /** Parse a single declaration. */
    bit parseDeclaration (inout Declaration [] list)
    {
        Declaration add;
        Token next, prev;

    restart:
        prev = token;
        next = token;
        if (token.type == lcurlyToken)
            return parseBlockDeclaration (list);
        else if (token.type == importToken)
            add = parseImportDeclaration ();
        else if (token.type == structToken)
            add =  parseStructDeclaration ();
        else if (token.type == classToken)
            add = parseClassDeclaration ();
        else if (token.type == aliasToken)
            add = parseAliasDeclaration ();
        else if (isAttributeSpecifier ())
            return parseAttributeSpecifier (list);
        else if (token.type == semicolonToken)
        {
            nextToken ();
            goto restart;
        }
        else if (token.type == thisToken)
        {
            ConstructorDeclaration decl = new ConstructorDeclaration (this, token.mark);
            Symbol [] args;
            int varargs;

            nextToken ();
            if (!parseParameters (args, varargs))
                return false;
            decl.name = new Symbol ("this");
            decl.type = new FunctionType (VoidType.singleton, args);
            (cast (FunctionType) decl.type).varargs = cast (bit) varargs;
            if (!parseContracts (decl.inStatement, decl.outStatement, decl.outReturn, decl.bodyStatement))
                return false;
            add = decl;
        }
        else if (token.type == unittestToken) add = parseUnittestDeclaration ();
        else if (token.type == enumToken)
            add = parseEnumDeclaration ();
        else if (isType (next, 2, reservedToken))
        {
            if (isFunctionDeclarationStart (next.type))
                add = parseFunctionDeclaration ();
            else if (isVariableDeclarationStart (next.type))
                add = parseVariableDeclaration ();
            else
                goto error;
        }
        else
        {
        error:
            parseError (new UnhandledTokenError (token.mark, fmt ("Did not expect %.*s when parsing declarations.", prev.typeName ())));
            nextToken ();
            return false;
        }

        if (add === null)
            return false;
        list ~= add;
        return true;
    }

    /** Parse a "unittest" function declaration. */
    Declaration parseUnittestDeclaration ()
    {
        UnittestDeclaration decl = new UnittestDeclaration (this, token.mark);
        Symbol [] args;

        if (!expectToken (unittestToken))
            return null;
        decl.name = new Symbol ("unittest");
        decl.type = new FunctionType (VoidType.singleton, args);
        if ((decl.bodyStatement = parseBlockStatement ()) === null)
            return null;
        return decl;
    }

    /** Parse a curly-bracket set of declarations. */
    bit parseBlockDeclaration (inout Declaration [] list)
    {
        if (!expectToken (lcurlyToken))
            return false;
        while (!handleToken (rcurlyToken))
        {
            if (!parseDeclaration (list))
                return false;
        }
        return true;
    }

    /** Parse an "enum" declaration. */
    EnumDeclaration parseEnumDeclaration ()
    {
        EnumDeclaration decl = new EnumDeclaration ();

        if (!expectToken (enumToken))
            return null;
        if (token.type == idToken)
        {
            decl.name = new Symbol (token.string);
            nextToken ();
        }

        if (handleToken (colonToken))
        {
            Symbol id;

            if ((decl.type = parseType (id, 0, lcurlyToken)) === null)
                return null;
        }

        if (!expectToken (lcurlyToken))
            return null;

        while (1)
        {
            Symbol item;

            if (handleToken (rcurlyToken))
                break;
            item = new Symbol ();
            if (!ensureToken (idToken))
                return null;
            item.name = token.string;
            item.type = decl.type;
            nextToken ();
            if (handleToken (assignToken))
            {
                if ((item.init = parseAssignExp ()) === null)
                    return null;
            }

            decl.list ~= item;
            if (!handleToken (commaToken) && !ensureToken (rcurlyToken))
                return null;
        }

        return decl;
    }

    /** Return whether a variable declaration follows. */
    bit isVariableDeclaration ()
    {
        Token next = token;

        if (!isType (next, 2, reservedToken))
            return false;
        return isVariableDeclarationStart (next.type);
    }

    /** Return whether this is the conceivable start of a variable declaration, after the type. */
    bit isVariableDeclarationStart (TokenType type)
    {
        return type == assignToken || type == commaToken || type == semicolonToken;
    }

    /** Parse a variable declaration. */
    VariableDeclaration parseVariableDeclaration ()
    {
        VariableDeclaration decl = new VariableDeclaration (this);
        Symbol id;
        Type type = parseType (id, 2, reservedToken);
        Marker start = token.mark;

        if (type === null)
            return null;

        while (1)
        {
            if (token.type == lbracketToken)
            {
                int depth = 0;

                parseError (new PostNameBracketError (token.mark));

                while (1)
                {
                    if (token.type == lbracketToken)
                        depth ++;
                    else if (token.type == rbracketToken)
                    {
                        depth --;
                        if (depth == 0)
                            break;
                    }
                    else if (token.type == eoiToken)
                    {
                        parseError (new UnhandledTokenError (start, "Post-name bracket never closed."));
                        return null;
                    }

                    nextToken ();
                }

                return null;
            }

            if (handleToken (assignToken))
            {
                if ((id.init = parseExpression ()) === null)
                    return null;
            }

            id.type = type;
            decl.list ~= id;

            if (handleToken (semicolonToken))
                break;
            if (!expectToken (commaToken))
                return null;
            if (!ensureToken (idToken))
                return null;
            id = new Symbol (token.string);
            id.name = token.string;
            nextToken ();
        }

        return decl;
    }

    bit isFunctionDeclaration ()
    {
        Token next = token;

        if (!isType (next, 2, reservedToken))
            return false;
        return isFunctionDeclarationStart (next.type);
    }

    bit isFunctionDeclarationStart (TokenType type)
    {
        return type == lcurlyToken || type == inToken || type == outToken || type == bodyToken;
    }

    FunctionDeclaration parseFunctionDeclaration ()
    {
        FunctionDeclaration decl = new FunctionDeclaration (this, token.mark);

        if ((decl.type = parseType (decl.name, 2, reservedToken)) === null)
            return null;
        if (!parseContracts (decl.inStatement, decl.outStatement, decl.outReturn, decl.bodyStatement))
            return null;
        return decl;
    }

    /* Parse a function body with contracts. */
    bit parseContracts (inout Statement inStatement, inout Statement outStatement, inout Symbol outReturn, inout Statement bodyStatement)
    {
        while (1)
        {
            switch (token.type)
            {
                case inToken:
                    nextToken ();
                    if (inStatement)
                    {
                        parseError (new MarkedError (token.mark, fmt ("Multiple \"in\" contracts given for function.")));
                        return false;
                    }

                    if ((inStatement = parseBlockStatement ()) === null)
                        return false;
                    continue;

                case outToken:
                    if (outStatement)
                    {
                        parseError (new MarkedError (token.mark, fmt ("Multiple \"out\" contracts given for function.")));
                        return false;
                    }

                    if (token.type == lparenToken)
                    {
                        nextToken ();
                        if (!ensureToken (idToken))
                            return false;
                        outReturn = new Symbol (token.string);
                        nextToken ();
                        if (!expectToken (rparenToken))
                            return false;
                    }

                    if ((outStatement = parseBlockStatement ()) === null)
                        return false;
                    continue;

                case bodyToken:
                    nextToken ();
                case lcurlyToken:
                    if (!ensureToken (lcurlyToken))
                        return false;
                    if ((bodyStatement = parseBlockStatement ()) === null)
                        return false;
                    return true;

                default:
                    parseError (new UnhandledTokenError (token.mark, fmt ("Did not expect %.*s while parsing function body.", token.quoted ())));
                    return false;
            }
        }
    }

    /** Return whether this is the start of a type, field, local, or function definition.
      * "needId" should be zero if this doesn't require an identifier, one if it can take
      * or leave an identifier, or two if it must have an identifier; the identifier is
      * stored in "id".  "endToken" indicates which token it must stop reading on,
      * and is generally reservedToken.
      */

    bit isType (inout Token token, int needId, TokenType endToken, out Token id)
    {
        int haveId;

        if (!isBasicType (token))
            return false;
        if (!isDeclarator (token, haveId, endToken))
            return false;
        if (needId == 1
         || (needId == 0 && !haveId)
         || (needId == 2 && haveId))
        {
            if (haveId)
                id = token;
            return true;
        }
        return false;
    }

    /** Form of isType without the identifier. */
    bit isType (inout Token token, int needId, TokenType endToken)
    {
        Token id;
        return isType (token, needId, endToken, id);
    }

    Type parseType (out Symbol id, int needId, TokenType endToken)
    {
        Marker base = token.mark;
        Type type;

        if ((type = parseBasicType ()) === null)
            return null;
        if (!parseDeclarator (type, id, endToken))
            return null;
        if (id !== null)
            id.type = type;
        if (needId == 1)
            return type;
        if (needId == 0 && id !== null)
        {
            parseError (new UnexpectedIdentifierError (base.contain (token.mark), id.toString ()));
            return null;
        }
        else if (needId == 2 && id === null)
        {
            parseError (new ExpectedIdentifierError (base.contain (token.mark)));
            return null;
        }
        return type;
    }

    Type parseBasicType ()
    {
        Token u;
        Type type;

        if (token.isBasicType ())
        {
            type = Type.findBasicType (token.string);
            nextToken ();
        }
        else switch (token.type)
        {
            case idToken:
            {
                char [] [] name;

                while (1)
                {
                    name ~= token.string;
                    nextToken ();
                    if (!handleToken (dotToken))
                        break;
                    if (!ensureToken (idToken))
                        return null;
                }

                return new SymbolType (name);
            }

            default:
                throw new Error (token.mark.message () ~ token.typeName ());
        }

        return type;
    }

    /** Return whether this is a basic type. */
    bit isBasicType (inout Token token)
    {
        Token t = token, u;

        if (t.isBasicType ())
            t = t.peek ();
        else switch (t.type)
        {
            case idToken:
            periodLabel:
                while (1)
                {
                    t = t.peek ();
                    if (t.type == dotToken)
                    {
                        t = t.peek ();
                        if (t.type != idToken)
                            goto falseLabel;
                    }
                    else
                        break;
                }
                break;

            case instanceToken:
                // Handle:
                //     instance Foo (int).bar x;
                // But not:
                //     instance Foo (int) x;
                // As it is an AliasDeclaration declaration.

                t = t.peek ();
                if (t.type != idToken)
                    goto falseLabel;
                t = t.peek ();
                while (t.type == dotToken)
                {
                    t = t.peek ();
                    if (t.type != idToken)
                        goto falseLabel;
                    t = t.peek ();
                }

                if (t.type != lparenToken)
                    goto falseLabel;

                /* Skip over the template arguments. */
                while (1)
                {
                    int parenCount = 0;

                    while (1)
                    {
                        t = t.peek ();
                        switch (t.type)
                        {
                            case lparenToken:
                                parenCount ++;
                                continue;
                            case rparenToken:
                                if (parenCount -- == 0)
                                    break;
                                continue;
                            case commaToken:
                                if (parenCount)
                                    continue;
                                break;
                            case eoiToken:
                            case semicolonToken:
                                goto falseLabel;
                            default:
                                continue;
                        }
                        break;
                    }

                    if (t.type != commaToken)
                        break;
                }

                if (t.type != rparenToken)
                    goto falseLabel;

                u = t.peek ();

                /* If it has a trailing "." it is a type. */
                if (u.type != dotToken)
                    goto falseLabel;
                goto periodLabel;

            default:
                goto falseLabel;
        }

        token = t;
        return true;

    falseLabel:
        return false;
    }

    /** Determine whether this is a declarator.  token is set to the first token afterwards, if true.
      * haveId is set to 0 or 1 depending upon whether an identifier was found.  endToken is the
      * token the parsing must abort on.
      */

    bit isDeclarator (inout Token token, inout int haveId, TokenType endToken)
    {
        Token t = token;
        bit parens;

        if (t.type == assignToken)
            return false;

        while (1)
        {
            parens = false;
            switch (t.type)
            {
                case mulToken:
                case andToken:
                    t = t.peek ();
                    continue;

                case lbracketToken: /* [ ], [ type ], [ expression ] */
                    t = t.peek ();
                    if (t.type == rbracketToken) /* [ ] */
                        t = t.peek ();
                    else if (isType (t, 0, rbracketToken, t)) /* [ type ] */
                        t = t.peek ();
                    else /* [ expression ] */
                    {
                        if (!isExpression (t))
                            return false;
                        if (t.type != rbracketToken)
                            return false;
                        t = t.peek ();
                    }
                    continue;

                case idToken:
                    if (haveId)
                        return false;
                    haveId = true;
                    t = t.peek ();
                    break;

                case lparenToken:
                    t = t.peek ();
                    if (t.type == rparenToken)
                        return false;
                    if (!isDeclarator (t, haveId, rparenToken))
                        return false;
                    t = t.peek ();
                    parens = true;
                    break;

                case delegateToken:
                    t = t.peek ();
                    if (!isParameters (t))
                        return false;
                    continue;

                default:
                    break;
            }
            break;
        }

        while (1)
        {
            switch (t.type)
            {
                case lbracketToken:
                    parens = false;
                    t = t.peek ();
                    if (t.type == rbracketToken) /* [ ] */
                        t = t.peek ();
                    else if (isType (t, 0, rbracketToken, t)) /* [ type ] */
                        t = t.peek ();
                    else /* [ expression ] */
                    {
                        if (!isExpression (t))
                            return false;
                        if (t.type != rbracketToken)
                            return false;
                        t = t.peek ();
                    }
                    continue;

                case lparenToken:
                    parens = false;
                    if (!isParameters (t))
                        return false;
                    continue;

                case rparenToken:
                case rbracketToken:
                case assignToken:
                case commaToken:
                case semicolonToken:
                case lcurlyToken:
                    if (!parens && (endToken == reservedToken || endToken == t.type))
                    {
                        token = t;
                        return true;
                    }
                    return false;

                default:
                    return false;
            }
        }
    }

    bit parseDeclarator (inout Type type, inout Symbol id, TokenType endToken)
    {
        Token next;
        Type ts, ta;
        bit parens = false;

        if (token.type == assignToken)
            return false;

        while (1)
        {
            parens = false;
            switch (token.type)
            {
                case mulToken:
                    nextToken ();
                    type = new PointerType (type);
                    continue;

                case lbracketToken: /* [ ], [ type ], [ expression ] */
                    nextToken ();
                    next = token;
                    if (token.type == rbracketToken) /* [ ] */
                    {
                        type = new ArrayType (type);
                        nextToken ();
                    }
                    else if (isType (next, 0, rbracketToken, token)) /* [ type ] */
                    {
                        Type sub;
                        Symbol id;

                        type = new AArrayType (type, sub = parseType (id, 0, rbracketToken));
                        if (sub === null)
                            return false;
                        nextToken ();
                    }
                    else /* [ expression ] */
                    {
                        Expression e;

                        type = new SArrayType (type, e = parseExpression ());
                        if (e === null)
                            return false;
                        if (!expectToken (rbracketToken))
                            return false;
                    }

                    continue;

                case idToken:
                    if (id !== null)
                    {
                        parseError (new UnhandledTokenError (token.mark, fmt ("Read multiple identifiers while parsing type, last is %.*s", token.quoted ())));
                        return false;
                    }

                    id = new Symbol ();
                    id.name = token.string;
                    nextToken ();
                    break;

                default:
                    break;
            }
            break;
        }

        while (1)
        {
            switch (token.type)
            {
                case lparenToken:
                {
                    Symbol [] args;
                    int varargs;

                    if (!parseParameters (args, varargs))
                        return false;
                    type = new FunctionType (type, args);
                    (cast (FunctionType) type).varargs = cast (bit) varargs;
                    continue;
                }

                case rparenToken:
                case rbracketToken:
                case assignToken:
                case commaToken:
                case semicolonToken:
                case lcurlyToken:
                    if (parens)
                    {
                        parseError (new UnhandledTokenError (token.mark, fmt ("Expected closing parentheses, not %.*s", token.quoted ())));
                        return false;
                    }

                    if (endToken == reservedToken || endToken == token.type)
                        return true;

                    /* fallthrough */
                default:
                    parseError (new UnhandledTokenError (token.mark, fmt ("%.*s was not expected when parsing declarator.", token.quoted ())));
                    return false;
            }
        }
    }

    bit isParameters (inout Token token)
    {
        Token t = token;
        int temp;

        if (t.type != lparenToken)
            return false;
        t = t.peek ();
        while (1)
        {
            switch (t.type)
            {
                case rparenToken:
                    break;

                case dotDotDotToken:
                    t = t.peek ();
                    break;

                case inToken:
                case outToken:
                case inoutToken:
                    t = t.peek ();

                default:
                    if (!isBasicType (t))
                        return false;
                    temp = false;
                    if (!isDeclarator (t, temp, reservedToken))
                        return false;
                    if (t.type == commaToken)
                    {
                        t = t.peek ();
                        continue;
                    }
                    break;
            }
            break;
        }

        if (t.type != rparenToken)
            return false;
        token = t.peek ();
        return true;
    }

    /** Parse a parameters list, return success. */
    bit parseParameters (inout Symbol [] list, out int varargs)
    {
        bit isin = true, isout, isinout;

        varargs = 0;
        if (!expectToken (lparenToken))
            return false;
        while (1)
        {
            Symbol arg;
            Type type;

            isin = true, isout = false, isinout = false;

            switch (token.type)
            {
                case rparenToken:
                    nextToken ();
                    return true;

                case dotDotDotToken:
                    nextToken ();
                    varargs = 1;
                    if (!expectToken (rparenToken))
                        return false;
                    return true;

                case inToken:
                    nextToken ();
                    goto readarg;

                case outToken:
                    isin = false;
                    isout = true;
                    nextToken ();
                    goto readarg;

                case inoutToken:
                    isin = false;
                    isinout = true;
                    nextToken ();
                    goto readarg;

                default:
                readarg:
                    if ((type = parseType (arg, 1, reservedToken)) === null)
                        return false;
                    if (arg === null)
                        arg = new Symbol ();
                    arg.isin = isin;
                    arg.isout = isout;
                    arg.isinout = isinout;
                    arg.type = type;

                    list ~= arg;
                    if (token.type == commaToken)
                    {
                        nextToken ();
                        continue;
                    }

                    if (!expectToken (rparenToken))
                        return false;
                    return true;

            }
        }
    }

    /** Parse an arguments list. */
    bit parseArguments (inout Expression [] list)
    {
        if (!expectToken (lparenToken))
            return false;
        if (handleToken (rparenToken))
            return true;
        while (1)
        {
            Expression e = parseAssignExp ();

            if (e === null)
                return false;
            list ~= e;
            if (handleToken (rparenToken))
                return true;
            if (!expectToken (commaToken))
                return false;
        }
    }

    /** Return whether this is the start of an attribute specifier. */
    bit isAttributeSpecifier ()
    {
        switch (token.type)
        {
            case externToken:
            case finalToken:
            case staticToken:
            case abstractToken:
            case overrideToken:
            case constToken:
                return true;

            default:
                return false;
        }
    }

    /** Parse an attribute specifier, which is either "attribute declaration" or "attribute declaration... [else declaration...]" */
    bit parseAttributeSpecifier (inout Declaration [] list)
    {
        int count = list.length, end;
        Declaration.Convention convention;
        bit isFinal, isStatic, isAbstract, isOverride, isConst, isExtern;

        bit setConvention, setFinal, setStatic;
        bit setAbstract, setOverride, setConst, setExtern;

        while (1)
        {
            if (handleToken (externToken))
            {
                setConvention = true;
                convention = Declaration.D;
                setExtern = true;
                isExtern = true;
                
                if (handleToken (lparenToken))
                {
                    if (!ensureToken (idToken))
                        setConvention = false;
                    else if (token.string == "C")
                        convention = Declaration.C;
                    else if (token.string == "D")
                        convention = Declaration.D;
                    else if (token.string == "Windows")
                        convention = Declaration.Windows;
                    else if (token.string == "Pascal")
                        convention = Declaration.Pascal;
                    else
                    {
                        parseError (new UnknownCallingConventionError (token.mark, token.string));
                        setConvention = false;
                        /* Continue reading, since the stream might not be corrupted. */
                    }

                    nextToken ();
                    if (!expectToken (rparenToken))
                        return false;
                }
            }
            else if (handleToken (finalToken))
                setFinal = true, isFinal = true;
            else if (handleToken (staticToken))
                setStatic = true, isStatic = true;
            else if (handleToken (abstractToken))
                setAbstract = true, isAbstract = true;
            else if (handleToken (overrideToken))
                setOverride = true, isOverride = true;
            else if (handleToken (constToken))
                setConst = true, isConst = true;
            else
                break;
        }

        if (!parseDeclaration (list))
            return false;
        end = list.length;

        if (token.type == elseToken)
        {
            nextToken ();
            if (!parseDeclaration (list))
                return false;
        }

        for (int c = count; c < end; c ++)
        {
            Declaration item = list [c];

            if (setConvention) item.convention = convention;
            if (setFinal) item.isFinal = isFinal;
            if (setStatic) item.isStatic = isStatic;
            if (setAbstract) item.isAbstract = isAbstract;
            if (setOverride) item.isOverride = isOverride;
            if (setConst) item.isConst = isConst;
            if (setExtern) item.isExtern = isExtern;
        }

        return true;
    }
    
    /** Parse an alias declaration, returning it or null on failure. */
    AliasDeclaration parseAliasDeclaration ()
    {
        AliasDeclaration decl = new AliasDeclaration ();
        int [5] x;
        alias x y;

        if (!expectToken (aliasToken)
         || (decl.type = parseType (decl.name, 2, semicolonToken)) === null
         || !expectToken (semicolonToken))
            return null;
        return decl;
    }

    /** Parse a struct declaration, returning it or null on failure. */
    StructDeclaration parseStructDeclaration ()
    {
        StructDeclaration decl = new StructDeclaration ();
        Declaration [] list;

        if (!expectToken (structToken))
            return null;

        if (token.type == idToken)
        {
            if ((decl.name = parseSymbolName ()) === null)
                return null;
        }

        if (!expectToken (lcurlyToken)
         || !parseDeclarationList (list, rcurlyToken)
         || !expectToken (rcurlyToken))
            return null;

        decl.scope = new Scope (this);
        decl.scope.root = decl;
        decl.scope.list = (Statement []) list;
        for (int c; c < list.length; c ++)
            list [c].parent = decl;
        return decl;
    }

    /** Parse a class declaration, returning it or null on failure. */
    ClassDeclaration parseClassDeclaration ()
    {
        ClassDeclaration decl = new ClassDeclaration ();
        Declaration [] list;

        if (!expectToken (classToken))
            return null;

        if ((decl.name = parseSymbolName ()) === null)
            return null;

        if (handleToken (colonToken))
        {
            Symbol parent;

            decl.supertype = (parent = parseSymbolName ());
            if (parent === null)
                return null;
        }

        if (!expectToken (lcurlyToken)
         || !parseDeclarationList (list, rcurlyToken)
         || !expectToken (rcurlyToken))
            return null;

        decl.scope = new Scope (this);
        decl.scope.root = decl;
        decl.scope.list = (Statement []) list;
        for (int c; c < list.length; c ++)
            list [c].parent = decl;
        return decl;
    }

    /** Parse an import declaration, returning it or null on failure. */
    ImportDeclaration parseImportDeclaration ()
    {
        ImportDeclaration st = new ImportDeclaration ();

        if (!expectToken (importToken))
            return null;
        while (1)
        {
            Symbol name = parseSymbolName ();

            /* Check for success; abort if failed. */
            if (name === null)
            {
                parseSemicolon ();
                return null;
            }

            st.names ~= name; /* Append the name. */

            /* Semicolon found; finished. */
            if (token.type == semicolonToken)
            {
                nextToken ();
                return st;
            }

            /* Comma not found; abort. */
            if (!expectToken (commaToken))
            {
                parseSemicolon ();
                return null;
            }
        }
    }

    /** Parse a period-separated symbol name and return it or null on failure. */
    Symbol parseSymbolName ()
    {
        Symbol name = new Symbol ();

        if (!ensureToken (idToken))
            return null;
        name.name = token.string;
        while (1)
        {
            if (nextToken ().type != dotToken)
                return name;
            nextToken ();
            if (!ensureToken (idToken))
                return null;
            name.name ~= "." ~ token.string;
        }
    }

    /** Parse a single statement. */
    Statement parseStatement ()
    {
        Expression e;
        Token next;
        
        next = token;
        if (handleToken (ifToken))
        {
            Expression test;
            Statement onTrue, onFalse;

            if (!expectToken (lparenToken)
             || (test = parseExpression ()) === null
             || !expectToken (rparenToken)
             || (onTrue = parseStatement ()) === null
             || (handleToken (elseToken) && (onFalse = parseStatement ()) === null))
                return null;
            return new IfElseStatement (test, onTrue, onFalse);
        }
        else if (handleToken (returnToken))
        {
            if (handleToken (semicolonToken))
                return new ReturnStatement (null);
            e = parseExpression ();
            if (e === null || !expectToken (semicolonToken))
                return null;
            return new ReturnStatement (e);
        }
        else if (handleToken (whileToken))
        {
            Expression test;
            Statement content;

            if (!expectToken (lparenToken)
             || (test = parseExpression ()) === null
             || !expectToken (rparenToken)
             || (content = parseStatement ()) === null)
                return null;
            return new WhileStatement (test, content);
        }
        else if (token.type == lcurlyToken) return parseBlockStatement ();
        else if (token.type == aliasToken) return parseAliasDeclaration ();
        else if (token.type == switchToken) return parseSwitchStatement ();
        else if (token.type == caseToken) return parseCaseStatement ();
        else if (token.type == defaultToken) return parseDefaultStatement ();
        else if (token.type == throwToken) return parseThrowStatement ();
        else if (token.type == withToken) return parseWithStatement ();
        else if (isLabelStatement ()) return parseLabelStatement ();
        else if (token.type == forToken) return parseForStatement ();
        else if (token.type == volatileToken) return parseVolatileStatement ();
        else if (token.type == staticToken)
        {
            nextToken ();
            if (isType (next, 2, reservedToken)
             && isVariableDeclarationStart (next.type))
            {
                VariableDeclaration decl = parseVariableDeclaration ();

                if (decl === null)
                    return null;
                decl.isStatic = true;
                return decl;
            }
            else
                parseError (new UnhandledTokenError (token.mark, fmt ("Expected variable after \"static\", got %.*s.", token.typeName ())));
        }
        else if (isType (next, 2, reservedToken))
        {
            if (isFunctionDeclarationStart (next.type))
                return parseFunctionDeclaration ();
            if (isVariableDeclarationStart (next.type))
                return parseVariableDeclaration ();
            else
            {
                parseError (new UnhandledTokenError (next.mark, fmt ("Unhandled declaration using %.*s.", next.typeName ())));
                return null;
            }
        }
        else if (handleToken (semicolonToken))
            parseError (new SemicolonStatementError (next.mark));
        else
        {
            e = parseExpression ();
            if (e === null || !expectToken (semicolonToken))
                return null;
            return new ExpStatement (e);
        }
    }

    /** Parse a block statement. */
    BlockStatement parseBlockStatement ()
    {
        BlockStatement st = new BlockStatement (this);
        Statement add;

        if (!expectToken (lcurlyToken))
            return null;

        while (1)
        {
            if (handleToken (rcurlyToken))
                return st;

            add = parseStatement ();
            if (add === null)
                return null;
            st.scope.add (st, add);
        }

        return st;
    }

    /** Parse a "volatile" statement. */
    VolatileStatement parseVolatileStatement ()
    {
        Marker mark = token.mark;
        Statement content;

        if (!expectToken (volatileToken)
         || (content = parseStatement ()) === null)
            return null;
        return new VolatileStatement (mark, content);
    }

    /** Parse a "with" statement. */
    WithStatement parseWithStatement ()
    {
        Marker mark = token.mark;
        Expression object;
        Statement content;

        if (!expectToken (withToken)
         || !expectToken (lparenToken)
         || (object = parseExpression ()) === null
         || !expectToken (rparenToken)
         || (content = parseStatement ()) === null)
            return null;
        return new WithStatement (mark, object, content);
    }

    /** Parse a "for" statement. */
    ForStatement parseForStatement ()
    {
        Marker mark = token.mark;
        Statement init, content;
        Expression test, incr;

        if (!expectToken (forToken)
         || !expectToken (lparenToken)
         || (!handleToken (semicolonToken) && (init = parseStatement ()) === null)
         || (!handleToken (semicolonToken) && ((test = parseExpression ()) === null || !handleToken (semicolonToken)))
         || (!handleToken (rparenToken) && ((incr = parseExpression ()) === null || !handleToken (rparenToken)))
         || (content = parseStatement ()) === null)
            return null;
        return new ForStatement (mark, init, test, incr, content);
    }

    /** Return whether this is the start of a label statement. */
    bit isLabelStatement ()
    {
        return token.type == idToken && token.peek ().type == colonToken;
    }

    /** Parse a label statement. */
    LabelStatement parseLabelStatement ()
    {
        Marker mark = token.mark;
        char [] name = token.string;

        if (!expectToken (idToken)
         || !expectToken (colonToken))
            return null;
        return new LabelStatement (mark, new Symbol (name));
    }

    /** Parse a "throw" statement. */
    ThrowStatement parseThrowStatement ()
    {
        Marker mark = token.mark;
        Expression value;

        if (!expectToken (throwToken)
         || (value = parseExpression ()) === null
         || !expectToken (semicolonToken))
            return null;
        return new ThrowStatement (mark, value);
    }

    /** Parse a "case" statement. */
    CaseStatement parseCaseStatement ()
    {
        Marker mark = token.mark;
        Expression value;

        if (!expectToken (caseToken)
         || (value = parseExpression ()) === null
         || !expectToken (colonToken))
            return null;
        return new CaseStatement (mark, value);
    }

    /** Parse a "default" statement. */
    DefaultStatement parseDefaultStatement ()
    {
        Marker mark = token.mark;

        if (!expectToken (defaultToken)
         || !expectToken (colonToken))
            return null;
        return new DefaultStatement (mark);
    }

    /** Parse a "switch" statement. */
    SwitchStatement parseSwitchStatement ()
    {
        Expression value;
        Statement content;

        if (!expectToken (switchToken)
         || !expectToken (lparenToken)
         || (value = parseExpression ()) === null
         || !expectToken (rparenToken)
         || (content = parseStatement ()) === null)
            return null;
        return new SwitchStatement (value, content);
    }

    /** Attempt to determine whether this is the start of an expression by scanning until a closing
      * right bracket is found.
      */
    bit isExpression (inout Token token)
    {
        Token t = token;
        int nest = 0;

        for ( ; ; t = t.peek ())
        {
            switch (t.type)
            {
                case lbracketToken:
                    nest ++;
                    continue;

                case rbracketToken:
                    if (-- nest >= 0)
                        continue;
                    break;

                case eoiToken:
                    return false;

                default:
                    continue;
            }
            break;
        }

        token = t;
        return true;
    }

    /** Parse an expression.
      * 
      * AssignExp
      * AssignExp , Expression
      */

    Expression parseExpression ()
    {
        Expression a, b;

        a = parseAssignExp ();
        if (a === null)
            return null;
        while (handleToken (commaToken))
        {
            a = new CommaExp (a, b = parseAssignExp ());
            if (b === null)
                return null;
        }

        return a;
    }

    /** Parse an assignment expression.
      *
      * @code
      * ConditionalExpression
      * ConditionalExpression = AssignExpression
      * ConditionalExpression += AssignExpression
      * ConditionalExpression -= AssignExpression
      * ConditionalExpression *= AssignExpression
      * ConditionalExpression /= AssignExpression
      * ConditionalExpression %= AssignExpression
      * ConditionalExpression &= AssignExpression
      * ConditionalExpression |= AssignExpression
      * ConditionalExpression ^= AssignExpression
      * ConditionalExpression ~= AssignExpression
      * ConditionalExpression <<= AssignExpression
      * ConditionalExpression >>= AssignExpression
      * ConditionalExpression >>>= AssignExpression
      * @endcode
      */

    Expression parseAssignExp ()
    {
        Expression a, b;

        if ((a = parseConditionalExp ()) === null)
            return null;
        TokenType type = token.type;

        if (handleToken (assignToken))
            a = new AssignExp (a, type, b = parseAssignExp ());
        else if (handleToken (catAssToken))
            a = new CatAssignExp (a, b = parseAssignExp ());
        else if (handleToken (addAssToken))
            a = new AddAssignExp (a, b = parseAssignExp ());
        else if (handleToken (mulAssToken))
            a = new MulAssignExp (a, b = parseAssignExp ());
        else if (handleToken (divAssToken))
            a = new DivAssignExp (a, b = parseAssignExp ());
        else if (handleToken (modAssToken))
            a = new ModAssignExp (a, b = parseAssignExp ());
        else if (handleToken (andAssToken))
            a = new BitwiseAndAssignExp (a, b = parseAssignExp ());
        else if (handleToken (orAssToken))
            a = new BitwiseOrAssignExp (a, b = parseAssignExp ());
        else if (handleToken (xorAssToken))
            a = new BitwiseXorAssignExp (a, b = parseAssignExp ());
        else if (handleToken (subAssToken))
            a = new SubAssignExp (a, b = parseAssignExp ());
        else if (handleToken (lshiftAssToken))
            a = new LeftShiftAssignExp (a, b = parseAssignExp ());
        else if (handleToken (rshiftAssToken))
            a = new RightShiftAssignExp (a, b = parseAssignExp ());
        else if (handleToken (urshiftAssToken))
            a = new UnsignedRightShiftAssignExp (a, b = parseAssignExp ());
        else
            return a;

        if (b === null)
            return null;

        return a;
    }

    /** Parse a conditional expression.
      *
      * @code
      * OrOrExpression
      * OrOrExpression ? Expression : ConditionalExpression
      * @endcode
      */

    Expression parseConditionalExp ()
    {
        Expression a, b, c;

        if ((a = parseOrOrExp ()) === null)
            return null;
        Marker mark = token.mark;
        if (handleToken (questionToken))
        {
            if ((b = parseExpression ()) === null
             || !expectToken (colonToken)
             || (c = parseConditionalExp ()) === null)
                return null;
            return new ConditionalExp (mark, a, b, c);
        }
        return a;
    }

    /** Parse an "a || b" expression.
      *
      * @code
      * AndAndExpression
      * AndAndExpression || AndAndExpression
      * @endcode
      */

    Expression parseOrOrExp ()
    {
        Expression a, b;

        if ((a = parseAndAndExp ()) === null)
            return null;
        while (handleToken (orOrToken))
        {
            a = new LogicalOrExp (a, b = parseAndAndExp ());
            if (b === null)
                return null;
        }
        return a;
    }

    /** Parse an "a && b" expression.
      *
      * OrExp
      * OrExp && OrExp
      */

    Expression parseAndAndExp ()
    {
        Expression a, b;

        if ((a = parseOrExp ()) === null)
            return null;
        while (handleToken (andAndToken))
        {
            a = new LogicalAndExp (a, b = parseOrExp ());
            if (b === null)
                return null;
        }
        return a;
    }

    /** Parse an "a | b" expression.
      *
      * @code
      * XorExpression
      * XorExpression | XorExpression
      * @endcode
      */

    Expression parseOrExp ()
    {
        Expression a, b;

        if ((a = parseXorExp ()) === null)
            return null;
        while (1)
        {
            if (handleToken (orToken))
                a = new BitwiseOrExp (a, b = parseXorExp ());
            else
                return a;
            if (b === null)
                return null;
        }
    }

    /** Parse an "a ^ b" expression.
      * AndExp
      * AndExp ^ AndExp
      */

    Expression parseXorExp ()
    {
        Expression a, b;

        if ((a = parseAndExp ()) === null)
            return null;
        while (1)
        {
            if (handleToken (xorToken))
                a = new BitwiseXorExp (a, b = parseAndExp ());
            else
                return a;
            if (b === null)
                return null;
        }
    }

    /** Parse an "a & b" expression.
      * 
      * @code
      * EqualExpression
      * EqualExpression & EqualExpression
      * @endcode
      */

    Expression parseAndExp ()
    {
        Expression a, b;

        if ((a = parseEqualExp ()) === null)
            return null;
        while (handleToken (andToken))
        {
            a = new BitwiseAndExp (a, b = parseEqualExp ());
            if (b === null)
                return null;
        }
        return a;
    }

    /** Parse the equals expressions.
      *
      * @code
      * RelExpression
      * RelExpression == RelExpression
      * RelExpression != RelExpression
      * RelExpression === RelExpression
      * RelExpression !== RelExpression
      * @endcode
      */

    Expression parseEqualExp ()
    {
        Expression a, b;

        if ((a = parseRelExp ()) === null)
            return null;
        while (1)
        {
            TokenType type = token.type;

            if (handleToken (equalsToken) || handleToken (notEqualsToken))
                a = new EqualsExp (a, type, b = parseRelExp ());
            else if (handleToken (isToken) || handleToken (isNotToken))
                a = new IdentityEqualsExp (a, type, b = parseRelExp ());
            else
                return a;
            if (b === null)
                return null;
        }
    }

    /** Parse comparison expressions.
      *
      * @code
      * ShiftExpression
      * ShiftExpression < ShiftExpression
      * ShiftExpression <= ShiftExpression
      * ShiftExpression > ShiftExpression
      * ShiftExpression >= ShiftExpression
      * ShiftExpression !<>= ShiftExpression
      * ShiftExpression !<> ShiftExpression
      * ShiftExpression <> ShiftExpression
      * ShiftExpression <>= ShiftExpression
      * ShiftExpression !> ShiftExpression
      * ShiftExpression !>= ShiftExpression
      * ShiftExpression !< ShiftExpression
      * ShiftExpression !<= ShiftExpression
      * ShiftExpression in ShiftExpression
      * @endcode
      */

    Expression parseRelExp ()
    {
        Expression a, b;

        if ((a = parseShiftExp ()) === null)
            return null;
        while (1)
        {
            TokenType type = token.type;

            switch (token.type)
            {
                case lessToken:
                case lequalsToken:
                case greaterToken:
                case gequalsToken:
                case ueToken:
                case unordToken:
                case ulToken:
                case uleToken:
                case ugToken:
                case ugeToken:
                    nextToken ();
                    a = new CompareExp (a, type, b = parseShiftExp ());
                    break;

                case inToken:
                    nextToken ();
                    a = new InExp (a, b = parseShiftExp ());
                    break;

                default:
                    return a;
            }

            if (b === null)
                return null;
        }
    }

    /** Parse a shift expression.
      *
      * @code
      * AddExpression
      * AddExpression << AddExpression
      * AddExpression >> AddExpression
      * AddExpression >>> AddExpression
      * @endcode
      */

    Expression parseShiftExp ()
    {
        Expression a, b;

        if ((a = parseAddExp ()) === null)
            return null;
        while (1)
        {
            if (handleToken (lshiftToken))
                a = new LeftShiftExp (a, b = parseAddExp ());
            else if (handleToken (rshiftToken))
                a = new RightShiftExp (a, b = parseAddExp ());
            else if (handleToken (urshiftToken))
                a = new UnsignedRightShiftExp (a, b = parseAddExp ());
            else
                return a;
            if (b === null)
                return null;
        }
    }

    /** Parse addition, subtraction, and concatenate expressions.
      *
      * @code
      * MulExpression
      * MulExpression + MulExpression
      * MulExpression - MulExpression
      * MulExpression ~ MulExpression
      * @endcode
      */

    Expression parseAddExp ()
    {
        Expression a, b;

        if ((a = parseMulExp ()) === null)
            return null;
        while (1)
        {
            if (handleToken (addToken))
                a = new AddExp (a, b = parseMulExp ());
            else if (handleToken (subToken))
                a = new SubExp (a, b = parseMulExp ());
            else if (handleToken (catToken))
                a = new CatExp (a, b = parseMulExp ());
            else
                return a;
            if (b === null)
                return null;
        }
    }

    /** Parse multiplication, division, and modulus expressions.
      *
      * @code
      * UnaryExpression
      * UnaryExpression * UnaryExpression
      * UnaryExpression / UnaryExpression
      * UnaryExpression % UnaryExpression
      * @endcode
      */

    Expression parseMulExp ()
    {
        Expression a, b;

        if ((a = parseUnaryExp ()) === null)
            return null;
        while (1)
        {
            if (handleToken (mulToken))
                a = new MulExp (a, b = parseUnaryExp ());
            else if (handleToken (divToken))
                a = new DivExp (a, b = parseUnaryExp ());
            else if (handleToken (modToken))
                a = new ModExp (a, b = parseUnaryExp ());
            else
                return a;
            if (b === null)
                return null;
        }
    }

    /** Parse a prefixed unary expression.
      *
      * @code
      * PostfixExpression
      * & UnaryExpression
      * ++ UnaryExpression
      * -- UnaryExpression
      * * UnaryExpression
      * - UnaryExpression
      * + UnaryExpression
      * ! UnaryExpression
      * ~ UnaryExpression
      * delete UnaryExpression
      * new NewExpression
      * cast ( Type ) UnaryExpression
      * ( Type ) UnaryExpression
      * ( Type ) . Identifier
      * @endcode
      */

    Expression parseUnaryExp ()
    {
        Marker mark = token.mark;
        Expression a, b;

        if (handleToken (addAddToken))
            a = new PreIncExp (token.mark, b = parseUnaryExp ());
        else if (handleToken (subToken))
            a = new NegExp (token.mark, b = parseUnaryExp ());
        else if (handleToken (subSubToken))
            a = new PreDecExp (token.mark, b = parseUnaryExp ());
        else if (handleToken (notToken))
            a = new LogicalNotExp (token.mark, b = parseUnaryExp ());
        else if (handleToken (castToken))
        {
            Symbol id;
            Type type;

            if (!expectToken (lparenToken))
                return null;
            if ((type = parseType (id, 0, rparenToken)) === null)
                return null;
            mark.contain (token.mark);
            if (!expectToken (rparenToken))
                return null;
            return new CastExp (mark, type, parseUnaryExp ());
        }
        else if (handleToken (andToken))
        {
            a = parseUnaryExp ();
            if (a === null)
                return null;
            return new AddressExp (mark.contain (a.mark), a);
        }
        else if (handleToken (mulToken))
        {
            a = parseUnaryExp ();
            if (a === null)
                return null;
            return new PointerExp (mark.contain (a.mark), a);
        }
        else if (handleToken (newToken))
        {
            Expression [] args;
            Type type;

            type = parseBasicType ();
            while (handleToken (mulToken))
                type = new PointerType (type);
            if (handleToken (lbracketToken))
            {
                Expression e = parseAssignExp ();
                Expression [] args;
                Symbol id;

                mark = mark.contain (token.mark);
                if (e === null || !expectToken (rbracketToken))
                    return null;
                if (!parseDeclarator (type, id, semicolonToken))
                    return null;
                type = new ArrayType (type);
            }
            else if (token.type == lparenToken && !parseArguments (args))
                return null;
            return new NewExp (mark, type, args);
        }
        else
            return parsePostfixExp ();

        if (b === null)
            return null;
        return a;
    }

    /** Parse a postfix expression.
      * 
      * @code
      * PrimaryExpression
      * PostfixExpression . Identifier
      * PostfixExpression ++
      * PostfixExpression --
      * PostfixExpression ( ArgumentList )
      * PostfixExpression [ AssignExp ]
      * PostfixExpression [ AssignExp .. AssignExp ]
      * PostfixExpression [ .. AssignExp ]
      * PostfixExpression [ AssignExp .. ]
      * PostfixExpression [ .. ]
      * @endcode
      */

    Expression parsePostfixExp ()
    {
        Expression a;

        a = parsePrimaryExp ();
        while (1)
        {
            if (handleToken (lparenToken))
            {
                Expression [] args;
                Expression arg;

                if (!handleToken (rparenToken))
                {
                    while (1)
                    {
                        arg = parseAssignExp ();
                        if (arg === null)
                            return null;
                        args ~= arg;
                        if (handleToken (rparenToken))
                            break;
                        if (!expectToken (commaToken))
                            return null;
                    }
                }

                a = new CallExp (a, args);
            }
            else if (handleToken (lbracketToken))
            {
                Expression start, end;

                if (handleToken (dotDotToken))
                {
                    if (handleToken (rbracketToken))
                        a = new SliceExp (a, null, null);
                    else
                    {
                        end = parseAssignExp ();
                        if (end === null || !expectToken (rbracketToken))
                            return null;
                        a = new SliceExp (a, null, end);
                    }
                }
                else
                {
                    start = parseAssignExp ();
                    if (start === null)
                        return null;
                    if (handleToken (dotDotToken))
                    {
                        end = parseAssignExp ();
                        if (end === null || !expectToken (rbracketToken))
                            return null;
                        a = new SliceExp (a, start, end);
                    }
                    else if (expectToken (rbracketToken))
                        a = new IndexExp (a, start);
                    else
                        return null;
                }
            }
            else if (handleToken (dotToken))
            {
                if (!ensureToken (idToken))
                    return null;
                a = new DotIdExp (a, token.string);
                nextToken ();
            }
            else if (handleToken (addAddToken))
                a = new PostIncExp (a);
            else if (handleToken (subSubToken))
                a = new PostDecExp (a);
            else
                break;
        }

        return a;
    }

    /** Parse a primary expression (basic types and predefined constants).
      *
      * @code
      * Identifier
      * this
      * super
      * null
      * true
      * false
      * NumericLiteral
      * StringLiteral
      * AssertExpression
      * Type . Identifier
      * ( Expression )
      * [ ArrayItems ]
      * @endcode
      */

    Expression parsePrimaryExp ()
    {
        Marker mark = token.mark;
        Expression a;

        if (token.isBasicType () && token.peek ().type == dotToken)
        {
            Type at = Type.findBasicType (token.string);
            char [] id;

            mark = token.mark;
            nextToken ();
            expectToken (dotToken);
            if (!ensureToken (idToken))
                return null;
            id = token.string;
            nextToken ();
            return new TypeDotIdExp (mark, at, id);
        }

        if (handleToken (assertToken))
        {
            Expression test;

            if (!expectToken (lparenToken)
             || (test = parseExpression ()) === null
             || !expectToken (rparenToken))
                return null;
            return new AssertExp (mark, test);
        }

        if (handleToken (lparenToken))
        {
            Token next = token, snext = token;
            int haveId;

            if (isType (next, 0, rparenToken))
            {
                Marker start = token.mark;

                next = next.peek ();
                if (next.isBasicType ())
                    goto basicType;
                switch (next.type)
                {
                    case dotToken:
                    case addAddToken:
                    case subSubToken:
                    case notToken:
                    case catToken:
                    case deleteToken:
                    case newToken:
                    case lparenToken:
                    case idToken:
                    case thisToken:
                    case superToken:
                    case lintToken:
                    case luintToken:
                    case llongToken:
                    case lulongToken:
                    case lrealToken:
                    case limaginaryToken:
                    case nullToken:
                    case trueToken:
                    case falseToken:
                    case lstringToken:
                    case andToken:
                    case mulToken:
                    case subToken:
                    case addToken:
                    basicType:
                    {
                        Type type = parseBasicType ();
                        Symbol id;

                        if (!parseDeclarator (type, id, rparenToken))
                            return null;
                        expectToken (rparenToken);
                        if (token.type == dotToken)
                        {
                            nextToken ();
                            if (!expectToken (idToken))
                                return null;
                            a = new TypeDotIdExp (start.contain (token.mark), type, token.string);
                            nextToken ();
                        }
                        else
                        {
                            a = parseUnaryExp ();
                            if (a === null)
                                return null;
                            a = new CastExp (start.contain (token.mark), type, a);
                        }

                        return a;
                    }

                    default:
                        break;
                }
            }

            a = parseExpression ();
            if (a === null || !expectToken (rparenToken))
                return null;
            return a;
        }

        switch (token.type)
        {
            case lintToken:
                a = new IntegerExp (token.mark, token.integer, IntType.singleton);
                nextToken ();
                return a;

            case llongToken:
                a = new IntegerExp (token.mark, token.integer, LongType.singleton);
                nextToken ();
                return a;

            case lulongToken:
                a = new IntegerExp (token.mark, token.integer, ULongType.singleton);
                nextToken ();
                return a;

            case luintToken:
                a = new IntegerExp (token.mark, token.integer, UIntType.singleton);
                nextToken ();
                return a;

            case lrealToken:
                a = new FloatExp (token.mark, token.floating, DoubleType.singleton);
                nextToken ();
                return a;

            case limaginaryToken:
                a = new ImaginaryExp (token.mark, token.floating);
                nextToken ();
                return a;

            case lstringToken:
            {
                Marker mark = token.mark;
                char [] string = token.string;

                nextToken ();
                while (token.type == lstringToken)
                {
                    string ~= token.string;
                    mark = mark.contain (token.mark);
                    nextToken ();
                }

                return new StringExp (mark, string);
            }

            case idToken:
                a = new IdExp (token.mark, token.string);
                nextToken ();
                return a;

            case thisToken:
                a = new ThisExp (token.mark);
                nextToken ();
                return a;

            case superToken:
                a = new SuperExp (token.mark);
                nextToken ();
                return a;

            case nullToken:
                a = new NullExp (token.mark, null);
                nextToken ();
                return a;

            case trueToken:
                a = new TrueExp (token.mark);
                nextToken ();
                return a;

            case falseToken:
                a = new FalseExp (token.mark);
                nextToken ();
                return a;

            case lbracketToken:
            {
                Marker mark = token.mark;
                ArrayItem [] list;

                nextToken ();
                mark = mark.contain (token.mark);
                while (!handleToken (rbracketToken))
                {
                    ArrayItem item;

                    if ((item.index = parseAssignExp ()) === null
                     || (handleToken (colonToken) && (item.value = parseAssignExp ()) === null))
                        return null;
                    list ~= item;
                    if (!handleToken (commaToken) && !ensureToken (rbracketToken))
                        return null;
                    mark = mark.contain (token.mark);
                }

                return new ArrayExp (mark, list);
            }

            default:
                parseError (new UnhandledTokenError (token.mark, fmt ("Couldn't handle %.*s while parsing an expression.", token.typeName ())));
                return null;
        }
    }
}
